#!/usr/bin/env python

import time
import socket
import psycopg2 as db
import psycopg2.extras
from mx.DateTime import *

import pprint

import config

dbcon = db.connect(config.dsn)
dbcon.set_isolation_level(2)
cur = dbcon.cursor(cursor_factory = psycopg2.extras.DictCursor)

update   = False
registry = {}
calls    = {}

def get_config_value(key):
    """
    Get value from config option from DB.
    """

    cur.execute("SELECT value FROM config WHERE key = %s", (key,))
    result = cur.fetchone()
    if result:
        return result['value']

    raise ValueError('config-key not found: %s' % key)

#    def get_extension(self, channel):
#        """
#        Strip trailing random stuff of channel to get extension.
#        """
#        return channel[:channel.rfind('-')]

CHAN_IDLE    = 0
CHAN_INUSE   = 1
CHAN_RINGING = 2

def get_calleridname(callerid, calleridname):
    cur.execute("SELECT lastname, firstname, info FROM phonebook WHERE phonenumber = %s", (callerid,))
    if cur.rowcount:
        row = cur.fetchone()
        if row['lastname'] and row['firstname'] and row['info']:
            calleridname = "%s, %s" % (row['lastname'], row['firstname'])
        elif row['lastname']:
            calleridname = row['lastname']
        elif row['firstname']:
            calleridname = row['firstname']
        if row['info']:
            calleridname += " - %s" % row['info']
        return calleridname
    return calleridname

class Channels:
    channels = {}
    
    def __repr__(self):
        return self.channels.__repr__()

    def get_channel(self, channel):
        return self.channels.setdefault(channel, {'id': None,
                                                  'callerid': None,
                                                  'calleridname': None})

    def create(self, ev):
        chan = self.get_channel(ev['Channel'])
        chan['id']           = ev['Uniqueid']
        chan['callerid']     = ev['CallerID']
        chan['calleridname'] = get_calleridname(ev['CallerID'], ev['CallerIDName'])
        if ev['State'] == 'Ringing':
            chan['state'] = CHAN_RINGING
        elif ev['State'] == 'Ring':
            chan['state'] = CHAN_INUSE
        else:
            chan['state'] = CHAN_IDLE

    def set_callerid(self, ev):
        chan = self.get_channel(ev['Channel'])
        chan['callerid']     = ev['CallerID']
        chan['calleridname'] = get_calleridname(ev['CallerID'], ev['CallerIDName'])

    def hangup(self, ev):
        channel = ev['Channel']
        self.get_channel(channel)
        del self.channels[channel]

    def dial(self, src, dest):
        chan = self.get_channel(src)
        chan['state']    = CHAN_INUSE
        chan = self.get_channel(dest)
        chan['state']    = CHAN_RINGING

    def link(self, src, dest):
        #chan = self.get_channel(src)
        #chan['state']    = CHAN_INUSE
        chan = self.get_channel(dest)
        chan['state']    = CHAN_INUSE
        

CALL_RINGING = 0
CALL_TALKING = 1
CALL_HANGUP  = 2

class Calls:
    calls = {}

    def __repr__(self):
        return self.calls.__repr__()

    def get_call(self, destid):
        return self.calls.setdefault(destid, {})
            
    def create(self, ev):
        global update

        src    = ev['Source']
        dest   = ev['Destination']
        destid = ev['DestUniqueID']

        call = self.get_call(destid)
        channels.dial(src, dest)
        call['stime'] = now()
        call['state'] = CALL_RINGING

        src_chan  = channels.get_channel(src)
        dest_chan = channels.get_channel(dest)
        cur.execute("INSERT INTO history (src, dest, stime, callerid, calleridname, state, destid)"
                    " VALUES (%s, %s, now(), %s, %s, %s, %s)",
                    (src, dest, src_chan['callerid'], src_chan['calleridname'], call['state'], destid))
        update = True

    def link(self, ev):
        global update

        src    = ev['Channel1']
        dest   = ev['Channel2']
        destid = ev['Uniqueid2']
        call = self.get_call(destid)
        channels.link(src, dest)
        call['ptime'] = now()
        call['state'] = CALL_TALKING

        cur.execute("UPDATE history SET ptime = now(), state = %s WHERE destid = %s",
                    (call['state'], destid))
        update = True

    def unlink(self, ev):
        global update

        src    = ev['Channel1']
        dest   = ev['Channel2']
        destid = ev['Uniqueid2']
        call = self.get_call(destid)
        call['etime'] = now()
        call['state'] = CALL_HANGUP

        cur.execute("UPDATE history SET etime = now(), state = %s WHERE destid = %s",
                    (call['state'], destid))
        update = True

    def hangup(self, ev):
        global update

        destid = ev['Uniqueid']
        if self.calls.has_key(destid):    # Try to hang up if channel is a valid destination
            call = self.get_call(destid)
            call['etime'] = now()
            call['state'] = CALL_HANGUP

            cur.execute("UPDATE history SET etime = now(), state = %s WHERE destid = %s AND etime IS NULL",
                        (call['state'], destid))
            del self.calls[destid]
            if cur.rowcount:
                update = True
        channels.hangup(ev)
            


class AMI:
    """
    Asterisk Manager Interface class
    """
    fd = None
    buffer = ''

    def __init__(self):
        self.login()

    def login(self):
        """
        Login to AMI. Get info from database.
        """
        ami_host = get_config_value('ami_host')
        ami_port = int(get_config_value('ami_port'))

        try:
            s = socket.socket(socket.AF_INET, socket.SOCK_STREAM)
            s.connect((ami_host, ami_port))
            self.fd = s.makefile()
        except socket.error, e:
            print "Could not connect to AMI: %s:%s"
            raise e

        # Get 'Asterisk Call Manager/1.0'
        ret = self.fd.readline()
        if ret != 'Asterisk Call Manager/1.0\r\n':
            raise Exception("Did not receive Asterisk Call Manager greeting.")

        ami_login = get_config_value('ami_login')
        ami_password = get_config_value('ami_password')

        self.fd.write('Action: Login\r\nUsername: %s\r\nSecret: %s\r\nEvents: on\r\n\r\n' % (ami_login, ami_password))
        self.fd.flush()

        ev = self.get_event()
        if ev['Message'] != 'Authentication accepted':
            raise Exception("Asterisk Manager Interface - Authentication failed.")

    def get_event(self):
        """
        Wait for an AMI event, put it in a dictionary and return it.
        """
        ev = {}
        line = self.fd.readline()[:-2]  # Strip trailing '\r\n'
        while line != '':
            key, value = line.split(': ', 1)
            ev[key] = value
            line = self.fd.readline()[:-2]
        
        return ev


def mainloop(ami):
    global update

    cur.execute("DELETE FROM state WHERE key = 'last_update'");
    cur.execute("INSERT INTO state (key, value) VALUES ('last_update', now())");
    cur.execute("COMMIT")
    while 1:
        ev = ami.get_event()
        if ev['Event'] == 'Newexten':   # We don't need this.
            continue

        # Asterisk shutdown
        if ev['Event'] == 'Shutdown':
            print "Asterisk server shutdown. Sleeping 10 seconds."
            time.sleep(10)
            return

        # Outgoing SIP-Registrations
        if ev['Event'] == 'Registry':
            channel = ev['Channel']
            domain  = ev['Domain']
            if not registry.has_key(channel):
                registry[channel] = {}
            if not registry[channel].has_key(domain):
                registry[channel][domain] = {}
            registry[channel][domain] = ev['Status']
            continue

        #
        # Channel events
        #
        # Incoming calls
        if ev['Event'] == 'Newchannel':     # Permission: read = call
            channels.create(ev)
        # Change in CallerID
        elif ev['Event'] == 'Newcallerid':  # Permission: read = call
            channels.set_callerid(ev)
        # Channel hangup
        elif ev['Event'] == 'Hangup':       # Permission: read = call
            calls.hangup(ev)    # Might have to cancel call first
        #
        # Call events
        #
        # Dial event
        elif ev['Event'] == 'Dial':         # Rermission: read = call
            calls.create(ev)
        # Call connected
        elif ev['Event'] == 'Link':         # Rermission: read = call
            calls.link(ev)
        # Call disconnected
        elif ev['Event'] == 'Unlink':       # Permission: read = call
            calls.unlink(ev)

        # Status update of (SIP) peers
        elif ev['Event'] == 'PeerStatus':   # Permission: read = system
            extension = ev['Peer']
            state = ev['PeerStatus']
            if state in ('Registered', 'Unregistered'):
                pass
                #hints.set_registration(extension, state)
            elif state in ('Reachable', 'Unreachable'):
                pass
                #hints.set_reachability(extension, state)

        if update:
            cur.execute("UPDATE state SET value = now() WHERE key = 'last_update'");
            cur.execute("COMMIT")
            update = False

        if config.debug:
            print "Event:", pprint.pformat(ev)
            print "Calls:", pprint.pformat(calls)
            print "Channels:", pprint.pformat(channels)
            print "Registry:", pprint.pformat(registry)
            
    

if __name__ == '__main__':
    while 1:
        try:
            ami      = AMI()
            channels = Channels()
            calls    = Calls()
            mainloop(ami)
        except KeyError, e:
            import traceback
            traceback.print_exc()
            time.sleep(10)
