/* pyid3lib: a python wrapper for the id3lib library for creating and
 * manipulating id3v1/v2 tags
 * Copyright 2000, 2003  Lars Bensmann (lars@almosthappy.de)
 *
 * This library is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Library General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at your
 * option) any later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Library General Public
 * License for more details.

 * You should have received a copy of the GNU Library General Public License
 * along with this library; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

 * You can download this software from:
 * http://www.almosthappy.de/pyid3lib/
 */

/******************************************************************************
* INSTANCE METHODS
******************************************************************************/

static PyObject * tag_Clear(PyObject *selfdummy, PyObject *args) {				/* tag.Clear() */
	tagobject* self;

	self = (tagobject *)selfdummy;
	if (!PyArg_ParseTuple(args, ""))
		return NULL;

	self->tag->Size();

	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject * tag_HasChanged(PyObject *selfdummy, PyObject *args) {			/* tag.HasChanged() */
	tagobject* self;
	bool retval;

	self = (tagobject *)selfdummy;
	if (!PyArg_ParseTuple(args, ""))
		return NULL;

	retval = self->tag->HasChanged();

	return Py_BuildValue("i", retval);
}

static PyObject * tag_Size(PyObject *selfdummy, PyObject *args) {				/* tag.Size() */
	tagobject* self;
	size_t retval;

	self = (tagobject *)selfdummy;
	if (!PyArg_ParseTuple(args, ""))
		return NULL;

	retval = self->tag->Size();

	return Py_BuildValue("l", retval);
}

static PyObject * tag_SetUnsync(PyObject *selfdummy, PyObject *args) {			/* tag.SetUnsync(bool) */
	int toggle;
	tagobject* self;
	bool retval;

	self = (tagobject *)selfdummy;
	if (!PyArg_ParseTuple(args, "i", &toggle))
		return NULL;
	retval = self->tag->SetUnsync(toggle);

	return Py_BuildValue("i", retval);
}

static PyObject * tag_SetExtendedHeader(PyObject *selfdummy, PyObject *args) {	/* tag.SetExtendedHeader(bool) */
	int toggle;
	tagobject* self;
	bool retval;

	self = (tagobject *)selfdummy;
	if (!PyArg_ParseTuple(args, "i", &toggle))
		return NULL;
	retval = self->tag->SetExtendedHeader(toggle);

	return Py_BuildValue("i", retval);
}

static PyObject * tag_SetPadding(PyObject *selfdummy, PyObject *args) {			/* tag.Padding(bool) */
	int toggle;
	tagobject* self;
	bool retval;

	self = (tagobject *)selfdummy;
	if (!PyArg_ParseTuple(args, "i", &toggle))
		return NULL;
	retval = self->tag->SetPadding(toggle);

	return Py_BuildValue("i", retval);
}

static PyObject * tag_AddFrame(PyObject *selfdummy, PyObject *args) {			/* tag.AddFrame(frame) */
	PyObject *object;
	tagobject* self;
	frameobject *frameobj;

	self = (tagobject *)selfdummy;
	if (!PyArg_ParseTuple(args, "O", &object))
		return NULL;
	if (!is_frameobject(object))
		onError("can only add a frame");
	frameobj = (frameobject *)object;
	Py_INCREF(frameobj);
	
	self->tag->AddFrame(frameobj->frame);

	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject * tag_AddFrames(PyObject *selfdummy, PyObject *args) {			/* tag.AddFrames([list,of,frames]) */

	onError("AddFrames not yet implemented");
}

static PyObject * tag_AttachFrame(PyObject *selfdummy, PyObject *args) {		/* tag.AttachFrame(frame) */
	PyObject *object;
	tagobject* self;
	frameobject *frameobj;

	self = (tagobject *)selfdummy;
	if (!PyArg_ParseTuple(args, "O", &object))
		return NULL;
	if (!is_frameobject(object))
		onError("can only attach a frame");
	frameobj = (frameobject *)object;
	
	Py_INCREF(frameobj);
	self->tag->AttachFrame(frameobj->frame);

	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject * tag_RemoveFrame(PyObject *selfdummy, PyObject *args) {		/* tag.RemoveFrame(frame) */
	PyObject *object;
	tagobject* self;
	frameobject *frameobj;

	self = (tagobject *)selfdummy;
	if (!PyArg_ParseTuple(args, "O", &object))
		return NULL;
	if (!is_frameobject(object))
		onError("can only remove a frame");
	frameobj = (frameobject *)object;
	
	self->tag->RemoveFrame(frameobj->frame);
	Py_DECREF(frameobj);

	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject * tag_Parse(PyObject *selfdummy, PyObject *args) {			/* tag.Parse(...) */

	onError("Parse not yet implemented");
}

static PyObject * tag_Link(PyObject *selfdummy, PyObject *args) {				/* tag.Link(filename) */
	char *filename;
	tagobject* self;
	size_t retval;

	self = (tagobject *)selfdummy;
	if (!PyArg_ParseTuple(args, "s", &filename))
		return NULL;
	retval = self->tag->Link(filename);

	return Py_BuildValue("l", retval);
}

static PyObject * tag_Update(PyObject *selfdummy, PyObject *args) {				/* tag.Update(flags = ID3TT_ALL) */
	int flags = ID3TT_ALL;
	tagobject* self;
	flags_t retval;

	self = (tagobject *)selfdummy;
	if (!(PyArg_ParseTuple(args, "i", &flags) || PyArg_ParseTuple(args, "")))
		return NULL;
	retval = self->tag->Update(flags);

	return Py_BuildValue("l", retval);
}

static PyObject * tag_Strip(PyObject *selfdummy, PyObject *args) {				/* tag.Strip(flags = ID3TT_ALL) */
	int flags = ID3TT_ALL;
	tagobject* self;
	flags_t retval;

	self = (tagobject *)selfdummy;
	if (!(PyArg_ParseTuple(args, "i", &flags) || PyArg_ParseTuple(args, "")))
		return NULL;
	retval = self->tag->Strip(flags);

	return Py_BuildValue("l", retval);
}

static PyObject * tag_Find(PyObject *selfdummy, PyObject *args) {				/* tag.Find(...) */
	ID3_FrameID frameid;
	ID3_FieldID fieldid;
	char *data;
	tagobject *self;
	ID3_Frame *frame;
	frameobject *frameobj;

	self = (tagobject *)selfdummy;
	if (PyArg_ParseTuple(args, "i", &frameid)) {
		frame = self->tag->Find(frameid);
	} else if (PyArg_ParseTuple(args, "iis", &frameid, &fieldid, &data)) {
		frame = self->tag->Find(frameid, fieldid, data);
	} else
		return NULL;

	if (frame) {
		frameobj = newframeobject();
		frameobj->frame = frame;
		return (PyObject *)frameobj;
	}

	Py_INCREF(Py_None);
	return Py_None;
}

static struct PyMethodDef tag_methods[] = {
 { "Clear",			tag_Clear,		1 },
 { "HasChanged",	tag_HasChanged,	1 },
 { "Size",			tag_Size,		1 },
 { "SetUnsync",		tag_SetUnsync,	1 },
 { "SetExtendedHeader",	tag_SetExtendedHeader,	1 },
 { "SetPadding",	tag_SetPadding,	1 },
 { "AddFrame",		tag_AddFrame,	1 },
 { "AddFrames",		tag_AddFrames,	1 },
 { "AttachFrame",	tag_AttachFrame,1 },
 { "RemoveFrame",	tag_RemoveFrame,1 },
 { "Parse",			tag_Parse,1 },
 { "Link",			tag_Link,		1 },
 { "Update",		tag_Update,		1 },
 { "Strip",			tag_Strip,		1 },
 { "Find",			tag_Find,		1 },
 { NULL,	NULL }
};

/*
 * vim:ts=4
 */
